<?php
/**
 * Copyright (c) 2002-2006 Aurlien Maille
 * 
 * This file is part of Wanewsletter.
 * 
 * Wanewsletter is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation; either version 2 
 * of the License, or (at your option) any later version.
 * 
 * Wanewsletter is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Wanewsletter; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 * @package Wanewsletter
 * @author  Bobe <wascripts@phpcodeur.net>
 * @link    http://phpcodeur.net/wascripts/wanewsletter/
 * @license http://www.gnu.org/copyleft/gpl.html  GNU General Public License
 * @version $Id: sqlite.php 423 2006-11-09 00:06:37Z bobe $
 */

if( !defined('_INC_CLASS_WADB_SQLITE') ) {

define('_INC_CLASS_WADB_SQLITE', true);

define('SQL_INSERT', 1);
define('SQL_UPDATE', 2);
define('SQL_DELETE', 3);

define('SQL_FETCH_NUM',   SQLITE_NUM);
define('SQL_FETCH_ASSOC', SQLITE_ASSOC);
define('SQL_FETCH_BOTH',  SQLITE_BOTH);

class Wadb_sqlite {
	
	/**
	 * Connexion  la base de donnes
	 * 
	 * @var resource
	 * @access private
	 */
	var $link;
	
	/**
	 * Nom de la base de donnes
	 * 
	 * @var string
	 * @access private
	 */
	var $dbname = '';
	
	/**
	 * Options de connexion
	 * 
	 * @var array
	 * @access private
	 */
	var $options = array();
	
	/**
	 * Code d'erreur
	 * 
	 * @var integer
	 * @access public
	 */
	var $errno = 0;
	
	/**
	 * Message d'erreur
	 * 
	 * @var string
	 * @access public
	 */
	var $error = '';
	
	/**
	 * Dernire requte SQL excute (en cas d'erreur seulement)
	 * 
	 * @var string
	 * @access public
	 */
	var $lastQuery = '';
	
	/**
	 * Nombre de requtes SQL excutes depuis le dbut de la connexion
	 * 
	 * @var integer
	 * @access public
	 */
	var $queries = 0;
	
	/**
	 * Dure totale d'excution des requtes SQL
	 * 
	 * @var integer
	 * @access public
	 */
	var $sqltime = 0;
	
	/**
	 * Version de la librairie SQLite
	 * 
	 * @var string
	 * @access public
	 */
	var $libVersion = '';
	
	/**
	 * Constructeur de classe
	 * 
	 * @param string $dbname   Nom de la base de donnes
	 * @param array  $options  Options de connexion/utilisation
	 * 
	 * @access public
	 */
	function Wadb_sqlite($sqlite_db, $options = null)
	{
		if( file_exists($sqlite_db) ) {
			if( !is_readable($sqlite_db) ) {
				trigger_error("SQLite database isn't readable!", E_USER_WARNING);
			}
		}
		else if( !is_writable(dirname($sqlite_db)) ) {
			trigger_error(dirname($sqlite_db) . " isn't writable. Cannot create "
				. basename($sqlite_db) . " database", E_USER_WARNING);
		}
		
		$connect = 'sqlite_open';
		
		if( is_array($options) ) {
			$this->options = $options;
			
			if( !empty($options['persistent']) ) {
				$connect = 'sqlite_popen';
			}
		}
		
		if( $this->link = $connect($sqlite_db, 0666, $this->error) ) {
			sqlite_exec($this->link, 'PRAGMA short_column_names = 1');
			sqlite_exec($this->link, 'PRAGMA case_sensitive_like = 0');
			
			ini_set('sqlite.assoc_case', '0');
			$this->libVersion = sqlite_libversion();
			
//			if( !empty($this->options['charset']) ) {
//				$this->encoding($this->options['charset']);
//			}
		}
	}
	
	/**
	 * Connexion  la base de donnes
	 * 
	 * @param array $infos    Informations de connexion
	 * @param array $options  Options de connexion/utilisation
	 * 
	 * @access public
	 * @return boolean
	 */
	function connect($infos = null, $options = null)
	{
		if( is_array($options) ) {
			$this->options = $options;
		}
		
		return true;
	}
	
	/**
	 * @access public
	 * @return boolean
	 */
	function isConnected()
	{
		return !is_null($this->link);
	}
	
	/**
	 * Renvoie le jeu de caractres courant utilis.
	 * Si l'argument $encoding est fourni, il est utilis pour dfinir
	 * le nouveau jeu de caractres de la connexion en cours
	 * 
	 * @param string $encoding
	 * 
	 * @access public
	 * @return string
	 */
	function encoding($encoding = null)
	{
		if( !is_null($encoding) ) {
			trigger_error("Setting encoding isn't supported by SQLite", E_USER_WARNING);
		}
		
		return sqlite_libencoding();
	}
	
	/**
	 * Excute une requte sur la base de donnes
	 * 
	 * @param string $query
	 * 
	 * @access public
	 * @return mixed
	 */
	function query($query)
	{
		$curtime = array_sum(explode(' ', microtime()));
		$result  = sqlite_query($this->link, $query);
		$endtime = array_sum(explode(' ', microtime()));
		
		$this->sqltime += ($endtime - $curtime);
		$this->queries++;
		
		if( !$result ) {
			$this->errno = sqlite_last_error($this->link);
			$this->error = sqlite_error_string($this->errno);
			$this->lastQuery = $query;
			
			$this->rollBack();
		}
		else {
			$this->errno = 0;
			$this->error = '';
			$this->lastQuery = '';
			
			if( !is_bool($result) ) {// on a rceptionn une ressource ou un objet
				$result = new WadbResult_sqlite($this->link, $result);
			}
		}
		
		return $result;
	}
	
	/**
	 * Construit une requte de type INSERT ou UPDATE  partir des diverses donnes fournies
	 * 
	 * @param string $type      Type de requte (peut valoir INSERT ou UPDATE)
	 * @param string $table     Table sur laquelle effectuer la requte
	 * @param array  $data      Tableau des donnes  insrer. Le tableau a la structure suivante:
	 *                          array(column_name => column_value[, column_name => column_value])
	 * @param array $sql_where  Chane de condition
	 * 
	 * @access public
	 * @return mixed
	 */
	function build($type, $table, $data, $sql_where = null)
	{
		$fields = $values = array();
		
		foreach( $data as $field => $value ) {
			if( is_null($value) ) {
				$value = 'NULL';
			}
			else if( is_bool($value) ) {
				$value = intval($value);
			}
			else if( !is_int($value) && !is_float($value) ) {
				$value = '\'' . $this->escape($value) . '\'';
			}
			
			array_push($fields, $this->quote($field));
			array_push($values, $value);
		}
		
		if( $type == SQL_INSERT ) {
			$query = sprintf('INSERT INTO %s (%s) VALUES(%s)', $table, implode(', ', $fields), implode(', ', $values));
		}
		else if( $type == SQL_UPDATE ) {
			
			$query = 'UPDATE ' . $table . ' SET ';
			for( $i = 0, $m = count($fields); $i < $m; $i++ ) {
				$query .= $fields[$i] . ' = ' . $values[$i] . ', ';
			}
			
			$query = substr($query, 0, -2);
			
			if( is_array($sql_where) && count($sql_where) > 0 ) {
				$query .= ' WHERE ';
				foreach( $sql_where as $field => $value ) {
					if( is_null($value) ) {
						$value = 'NULL';
					}
					else if( is_bool($value) ) {
						$value = intval($value);
					}
					else if( !is_int($value) && !is_float($value) ) {
						$value = '\'' . $this->escape($value) . '\'';
					}
					
					$query .= sprintf('%s = %s AND ', $this->quote($field), $value);
				}
				
				$query = substr($query, 0, -5);
			}
		}
		
		return $this->query($query);
	}
	
	/**
	 * Protge un nom de base, de table ou de colonne en prvision de son utilisation
	 * dans une requte
	 * 
	 * @param string $name
	 * 
	 * @access public
	 * @return string
	 */
	function quote($name)
	{
		return '[' . $name . ']';
	}
	
	/**
	 * @param mixed $tables  Nom de table ou tableau de noms de table
	 * 
	 * @access public
	 * @return void
	 */
	function vacuum($tables)
	{
		if( !is_array($tables) ) {
			$tables = array($tables); 
		}
		
		foreach( $tables as $tablename ) {
			sqlite_exec($this->link, 'VACUUM ' . $tablename);
		}
	}
	
	/**
	 * Dmarre le mode transactionnel
	 * 
	 * @access public
	 * @return boolean
	 */
	function beginTransaction()
	{
		return sqlite_exec($this->link, 'BEGIN');
	}
	
	/**
	 * Envoie une commande COMMIT  la base de donnes pour validation de la
	 * transaction courante
	 * 
	 * @access public
	 * @return boolean
	 */
	function commit()
	{
		if( !($result = sqlite_exec($this->link, 'COMMIT')) )
		{
			sqlite_exec($this->link, 'ROLLBACK');
		}
		
		return $result;
	}
	
	/**
	 * Envoie une commande ROLLBACK  la base de donnes pour annulation de la
	 * transaction courante
	 * 
	 * @access public
	 * @return boolean
	 */
	function rollBack()
	{
		return sqlite_exec($this->link, 'ROLLBACK');
	}
	
	/**
	 * Renvoie le nombre de lignes affectes par la dernire requte DML
	 * 
	 * @access public
	 * @return boolean
	 */
	function affectedRows()
	{
		return sqlite_changes($this->link);
	}
	
	/**
	 * Retourne l'identifiant gnr automatiquement par la dernire requte
	 * INSERT sur la base de donnes
	 * 
	 * @access public
	 * @return integer
	 */
	function lastInsertId()
	{
		return sqlite_last_insert_rowid($this->link);
	}
	
	/**
	 * chappe une chane en prvision de son insertion dans une requte sur
	 * la base de donnes
	 * 
	 * @param string $string
	 * 
	 * @access public
	 * @return string
	 */
	function escape($string)
	{
		return sqlite_escape_string($string);
	}
	
	/**
	 * Vrifie l'tat de la connexion courante et effectue si besoin une reconnexion
	 * 
	 * @access public
	 * @return boolean
	 */
	function ping()
	{
		return true;
	}
	
	/**
	 * Ferme la connexion  la base de donnes
	 * 
	 * @access public
	 * @return boolean
	 */
	function close()
	{
		if( !is_null($this->link) ) {
			@$this->rollBack();
			$result = sqlite_close($this->link);
			$this->link = null;
			
			return $result;
		}
		else {
			return true;
		}
	}
	
	/**
	 * Destructeur de classe
	 * 
	 * @access public
	 * @return void
	 */
	function __destruct()
	{
		$this->close();
	}
}

class WadbResult_sqlite {
	
	/**
	 * Connexion  la base de donnes
	 * 
	 * @var resource
	 * @access private
	 */
	var $link;
	
	/**
	 * Ressource de rsultat de requte
	 * 
	 * @var resource
	 * @access private
	 */
	var $result;
	
	/**
	 * Mode de rcupration des donnes
	 * 
	 * @var integer
	 * @access private
	 */
	var $fetchMode;
	
	/**
	 * Constructeur de classe
	 * 
	 * @param resource $link    Ressource de connexion  la base de donnes
	 * @param resource $result  Ressource de rsultat de requte
	 * 
	 * @access public
	 */
	function WadbResult_sqlite($link, $result)
	{
		$this->link   = $link;
		$this->result = $result;
		$this->fetchMode = SQLITE_BOTH;
	}
	
	/**
	 * Renvoie la ligne suivante dans le jeu de rsultat
	 * 
	 * @param integer $mode  Mode de rcupration des donnes
	 * 
	 * @access public
	 * @return array
	 */
	function fetch($mode = null)
	{
		if( is_null($mode) ) {
			$mode = $this->fetchMode;
		}
		
		return sqlite_fetch_array($this->result, $mode);
	}
	
	/**
	 * Renvoie sous forme d'objet la ligne suivante dans le jeu de rsultat
	 * 
	 * @access public
	 * @return object
	 */
	function fetchObject()
	{
		return sqlite_fetch_object($this->result);
	}
	
	/**
	 * Renvoie un tableau de toutes les lignes du jeu de rsultat
	 * 
	 * @param integer $mode  Mode de rcupration des donnes
	 * 
	 * @access public
	 * @return array
	 */
	function fetchAll($mode = null)
	{
		if( is_null($mode) ) {
			$mode = $this->fetchMode;
		}
		
		return sqlite_fetch_all($this->result, $mode);
	}
	
	/**
	 * Retourne le contenu de la colonne pour l'index ou le nom donn
	 *  l'index suivant dans le jeu de rsultat.
	 * 
	 * @param mixed $column  Index ou nom de la colonne
	 * 
	 * @access public
	 * @return string
	 */
	function column($column)
	{
		$row = sqlite_fetch_array($this->result);
		
		return (is_array($row) && isset($row[$column])) ? $row[$column] : false;
	}
	
	/**
	 * Configure le mode de rcupration par dfaut
	 * 
	 * @param integer $mode  Mode de rcupration des donnes
	 * 
	 * @access public
	 * @return boolean
	 */
	function setFetchMode($mode)
	{
		if( in_array($mode, array(SQLITE_NUM, SQLITE_ASSOC, SQLITE_BOTH)) ) {
			$this->fetchMode = $mode;
			return true;
		}
		else {
			trigger_error("Invalid fetch mode", E_USER_WARNING);
			return false;
		}
	}
	
	/**
	 * Libre la mmoire alloue
	 * 
	 * @access public
	 * @return void
	 */
	function free()
	{
		if( !is_null($this->result) ) {
			$this->result = null;
		}
	}
	
	/**
	 * Destructeur de classe
	 * 
	 * @access public
	 * @return void
	 */
	function __destruct()
	{
		$this->free();
	}
}

class WadbBackup_sqlite {
	
	/**
	 * Informations concernant la base de donnes
	 * 
	 * @var array
	 * @access private
	 */
	var $infos = array();
	
	/**
	 * Fin de ligne
	 * 
	 * @var boolean
	 * @access public
	 */
	var $eol = "\n";
	
	/**
	 * Constructeur de classe
	 * 
	 * @param array $infos  Informations concernant la base de donnes
	 * 
	 * @access public
	 */
	function WadbBackup_sqlite($infos)
	{
		$this->infos = $infos;
		
		if( !isset($this->infos['host']) ) {
			$this->infos['host'] = isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : 'Unknown';
		}
	}
	
	/**
	 * Gnration de l'en-tte du fichier de sauvegarde
	 * 
	 * @param string $toolname  Nom de l'outil utilis pour gnrer la sauvegarde
	 * 
	 * @access public
	 * @return string
	 */
	function header($toolname = '')
	{
		global $db;
		
		$contents  = '-- ' . $this->eol;
		$contents .= "-- $toolname SQLite Dump" . $this->eol;
		$contents .= '-- ' . $this->eol;
		$contents .= "-- Host       : " . $this->infos['host'] . $this->eol;
		$contents .= "-- SQLite lib : " . $db->libVersion . $this->eol;
		$contents .= "-- Database   : " . basename($this->infos['dbname']) . $this->eol;
		$contents .= '-- Date       : ' . date('d/m/Y H:i:s O') . $this->eol;
		$contents .= '-- ' . $this->eol;
		$contents .= $this->eol;
		
		return $contents;
	}
	
	/**
	 * Retourne la liste des tables prsentes dans la base de donnes considre
	 * 
	 * @access public
	 * @return array
	 */
	function get_tables()
	{
		global $db;
		
		if( !($result = $db->query("SELECT tbl_name FROM sqlite_master WHERE type = 'table'")) ) {
			trigger_error('Impossible d\'obtenir la liste des tables', ERROR);
		}
		
		$tables = array();
		while( $row = $result->fetch() ) {
			$tables[$row['tbl_name']] = '';
		}
		
		return $tables;
	}
	
	/**
	 * Utilisable pour l'ajout de requte supplmentaires (squences, configurations diverses, etc)
	 * 
	 * @param boolean $drop_option
	 * 
	 * @access public
	 * @return string
	 */
	function get_other_queries($drop_option)
	{
		return '';
	}
	
	/**
	 * Retourne la structure d'une table de la base de donnes sous forme de requte SQL de type DDL
	 * 
	 * @param array   $tabledata    Informations sur la table (provenant de self::get_tables())
	 * @param boolean $drop_option  Ajouter une requte de suppression conditionnelle de table
	 * 
	 * @access public
	 * @return string
	 */
	function get_table_structure($tabledata, $drop_option)
	{
		global $db;
		
		$contents  = '-- ' . $this->eol;
		$contents .= '-- Struture de la table ' . $tabledata['name'] . ' ' . $this->eol;
		$contents .= '-- ' . $this->eol;
		
		if( $drop_option ) {
			$contents .= 'DROP TABLE ' . $tabledata['name'] . ';' . $this->eol;
		}
		
		$sql = "SELECT sql, type
			FROM sqlite_master
			WHERE tbl_name = '$tabledata[name]'
				AND sql IS NOT NULL";
		if( !($result = $db->query($sql)) ) {
			trigger_error('Impossible d\'obtenir la structure de la table', ERROR);
		}
		
		$indexes = '';
		while( $row = $result->fetch() ) {
			if( $row['type'] == 'table' ) {
				$create_table = str_replace(',', ',' . $this->eol, $row['sql']) . ';' . $this->eol;
			}
			else {
				$indexes .= $row['sql'] . ';' . $this->eol;
			}
		}
		
		$contents .= $create_table . $indexes;
		
		return $contents;
	}
	
	/**
	 * Retourne les donnes d'une table de la base de donnes sous forme de requtes SQL de type DML
	 * 
	 * @param string $tablename  Nom de la table  considrer
	 * 
	 * @access public
	 * @return string
	 */
	function get_table_data($tablename)
	{
		global $db;
		
		$contents = '';
		
		$sql = 'SELECT * FROM ' . $tablename;
		if( !($result = $db->query($sql)) ) {
			trigger_error('Impossible d\'obtenir le contenu de la table ' . $tablename, ERROR);
		}
		
		$result->setFetchMode(SQL_FETCH_ASSOC);
		
		if( $row = $result->fetch() ) {
			$contents  = $this->eol;
			$contents .= '-- ' . $this->eol;
			$contents .= '-- Contenu de la table ' . $tablename . ' ' . $this->eol;
			$contents .= '-- ' . $this->eol;
			
			$fields = array();
			for( $j = 0, $n = sqlite_num_fields($result->result); $j < $n; $j++ ) {
				$fields[] = sqlite_field_name($result->result, $j);
			}
			
			$fields = implode(', ', $fields);
			
			do {
				$contents .= "INSERT INTO $tablename ($fields) VALUES";
				
				foreach( $row as $key => $value ) {
					if( is_null($value) ) {
						$row[$key] = 'NULL';
					}
					else {
						$row[$key] = '\'' . $db->escape($value) . '\'';
					}
				}
				
				$contents .= '(' . implode(', ', $row) . ');' . $this->eol;
			}
			while( $row = $result->fetch() );
		}
		
		return $contents;
	}
}

}
?>
